import os
import datetime
import json
from flask import Flask, request, render_template, jsonify, redirect, url_for
from flask_cors import CORS
from supabase import create_client
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# Supabase Configuration
SUPABASE_URL = os.environ.get("SUPABASE_URL", "https://rmccnszlwmcrjolzfokk.supabase.co")
SUPABASE_KEY = os.environ.get("SUPABASE_KEY", "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6InJtY2Nuc3psd21jcmpvbHpmb2trIiwicm9sZSI6InNlcnZpY2Vfcm9sZSIsImlhdCI6MTczODYzODE3OSwiZXhwIjoyMDU0MjE0MTc5fQ.zEPSqh1vdfZ2fGLJYZJ9luzFgnjn8RUY9jkr8f59_T4")

# Initialize Flask app
app = Flask(__name__)
CORS(app)  # Enable CORS for all routes

# Initialize Supabase client
try:
    supabase = create_client(SUPABASE_URL, SUPABASE_KEY)
    print("✅ Supabase connection for QR Scanner established")
except Exception as e:
    print(f"❌ Error connecting to Supabase: {e}")
    supabase = None

@app.route('/')
def index():
    """Main page with QR scanner"""
    return render_template('index.html')

@app.route('/stats')
def stats():
    """Statistics page for event organizers"""
    return render_template('stats.html')

@app.route('/api/verify/<uuid:unique_id>', methods=['GET'])
def verify_qr(unique_id):
    """Verify QR code and record attendance"""
    if not supabase:
        return jsonify({"error": "Database connection not available"}), 500
    
    try:
        # Convert UUID to string for database query
        unique_id_str = str(unique_id)
        
        # Query the database for the participant
        response = supabase.table("rsvp_submissions").select("*").eq("id", unique_id_str).execute()
        
        if not response.data or len(response.data) == 0:
            return jsonify({"error": "QR code not recognized", "success": False}), 404
        
        participant = response.data[0]
        
        # Check if already attended
        already_attended = participant.get("attended", False)
        
        # Record attendance if not already marked
        current_time = datetime.datetime.now(datetime.timezone.utc).isoformat()
        
        update_data = {
            "attended": True,
            "attendance_time": current_time
        }
        
        supabase.table("rsvp_submissions").update(update_data).eq("id", unique_id_str).execute()
        
        # Return participant details and attendance status
        return jsonify({
            "success": True,
            "participant": {
                "name": participant.get("full_name"),
                "email": participant.get("email"),
                "organization": participant.get("organization"),
                "role": participant.get("role"),
                "already_attended": already_attended,
                "attendance_time": current_time
            }
        })
    
    except Exception as e:
        print(f"❌ Error verifying QR code: {e}")
        return jsonify({"error": f"Error processing QR code: {str(e)}", "success": False}), 500

@app.route('/api/stats', methods=['GET'])
def get_stats():
    """Get attendance statistics"""
    if not supabase:
        return jsonify({"error": "Database connection not available"}), 500
    
    try:
        # Get all registrations
        registered = supabase.table("rsvp_submissions").select("*").execute()
        
        # Get all attended participants
        attended = supabase.table("rsvp_submissions").select("*").eq("attended", True).execute()
        
        # Calculate statistics
        total_registered = len(registered.data)
        total_attended = len(attended.data)
        attendance_rate = (total_attended / total_registered * 100) if total_registered > 0 else 0
        
        # Get list of recent check-ins (last 10)
        recent_checkins = sorted(
            [p for p in attended.data if p.get("attendance_time")],
            key=lambda x: x.get("attendance_time", ""),
            reverse=True
        )[:10]
        
        # Format recent check-ins for display
        formatted_checkins = []
        for checkin in recent_checkins:
            try:
                # Parse ISO format datetime
                attendance_time = datetime.datetime.fromisoformat(
                    checkin.get("attendance_time").replace("Z", "+00:00")
                )
                # Format to readable time
                formatted_time = attendance_time.strftime("%Y-%m-%d %H:%M:%S")
            except:
                formatted_time = checkin.get("attendance_time", "Unknown")
                
            formatted_checkins.append({
                "name": checkin.get("full_name", "Unknown"),
                "email": checkin.get("email", ""),
                "organization": checkin.get("organization", ""),
                "time": formatted_time
            })
            
        return jsonify({
            "success": True,
            "stats": {
                "total_registered": total_registered,
                "total_attended": total_attended,
                "attendance_rate": round(attendance_rate, 1),
                "recent_checkins": formatted_checkins
            }
        })
        
    except Exception as e:
        print(f"❌ Error getting statistics: {e}")
        return jsonify({"error": f"Error retrieving statistics: {str(e)}", "success": False}), 500

# if __name__ == '__main__':
#     app.run(debug=True, port=5002)  # Using port 5001 to avoid conflict with existing RSVP app
if __name__ == '__main__':
    app.run()